#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Módulo de imagens embutidas (embedded images)
Todas as imagens são armazenadas em base64 para não dependerem de arquivos externos.
"""

import base64
from io import BytesIO
from PIL import Image

# Imagens em base64

RISCO_PT_B64 = """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"""

RISCO_EN_B64 = """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"""

LOGO_B64 = """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"""

def get_risco_image_pt():
    """Retorna a imagem Risco.png em português como objeto PIL.Image"""
    try:
        image_bytes = base64.b64decode(RISCO_PT_B64)
        return Image.open(BytesIO(image_bytes))
    except Exception as e:
        print(f"❌ Erro ao carregar imagem Risco PT: {e}")
        return None

def get_risco_image_en():
    """Retorna a imagem Risk-Ingles.png em inglês como objeto PIL.Image"""
    try:
        image_bytes = base64.b64decode(RISCO_EN_B64)
        return Image.open(BytesIO(image_bytes))
    except Exception as e:
        print(f"❌ Erro ao carregar imagem Risco EN: {e}")
        return None

def get_risk_image(language='pt'):
    """Retorna a imagem de risco baseada no idioma (pt ou en)"""
    if language == 'en':
        return get_risco_image_en()
    else:
        return get_risco_image_pt()

def get_logo_image():
    """Retorna a imagem do logo FastTag como objeto PIL.Image"""
    try:
        image_bytes = base64.b64decode(LOGO_B64)
        return Image.open(BytesIO(image_bytes))
    except Exception as e:
        print(f"❌ Erro ao carregar logo: {e}")
        return None

def get_risk_image_bytes(language='pt'):
    """Retorna os bytes brutos da imagem de risco (útil para PDF)"""
    try:
        if language == 'en':
            return base64.b64decode(RISCO_EN_B64)
        else:
            return base64.b64decode(RISCO_PT_B64)
    except Exception as e:
        print(f"❌ Erro ao carregar bytes da imagem de risco: {e}")
        return None

def get_logo_image_bytes():
    """Retorna os bytes brutos do logo (útil para PDF)"""
    try:
        return base64.b64decode(LOGO_B64)
    except Exception as e:
        print(f"❌ Erro ao carregar bytes do logo: {e}")
        return None
